__all__ = (
    "get_attrs_of",
    "get_slots_of",
    "combine_classes",
    "copy_class_docs",
    "copy_docs",
)
import builtins
import functools
import types
from typing import Any, Iterable

_known_builtins = frozenset(
    v for k, v in vars(builtins).items() if not k.startswith("_")
)


def get_slots_of(kls: type) -> Iterable[tuple[type, None | tuple[str, ...]]]:
    """Visit a class MRO collecting all slotting

    This cannot collect slotting of C objects- python builtins like object,
    or literal python C extensions, not unless they expose __slots__.

    """
    for base in kls.mro():
        yield (
            base,
            # class objects provide a proxy map so abuse that to look at the class
            # directly.
            base.__dict__.get("__slots__", () if base in _known_builtins else None),
        )


def get_attrs_of(
    obj: Any, weakref=False, suppressions: Iterable[str] = (), _sentinel=object()
) -> Iterable[tuple[str, Any]]:
    """
    yield the attributes of a given instance.

    This handles both slotted and non slotted classes- slotted
    classes do not have __dict__.  It also handles mixed derivations,
    a non slotted class that inherited from a slotted class.

    For an ordered __dict__ class, the ordering is *not* honored in what
    this yields.

    :param weakref: by default, suppress that __weakref__ exists since it's
      python internal bookkeeping.  The only reason to enable this is for introspection
      tools; even state saving tools shouldn't care about __weakref__
    :param suppressions: attributes to suppress from the return.  Use this
      as a way to avoid having to write a filter in the consumer- this already
      has to do filtering after all.
    """
    seen = set(suppressions)
    seen.add("__weakref__")
    # if weakref is supported, it's either a ref or None- the attribute always
    # exists however.
    if weakref and (r := getattr(obj, "__weakref__", None)) is not None:
        yield "__weakref__", r
    for k, v in getattr(obj, "__dict__", {}).items():
        if k not in seen:
            yield (k, v)
            seen.add(k)
    for _, slots in get_slots_of(type(obj)):
        if slots is None:
            continue
        for slot in slots:
            if slot not in seen:
                if (o := getattr(obj, slot, _sentinel)) is not _sentinel:
                    yield slot, o
                    seen.add(slot)


@functools.lru_cache
def combine_classes(kls: type, *extra: type) -> type:
    """Given a set of classes, combine this as if one had wrote the class by hand

    This is primarily for composing metaclasses on the fly; this:

    class foo(metaclass=combine_metaclasses(kls1, kls2, kls3)): pass

    is the same as if you did this:

    class mkls(kls1, kls2, kls3): pass
    class foo(metaclass=mkls): pass
    """
    klses = [kls]
    klses.extend(extra)

    if len(klses) == 1:
        return kls

    class combined(*klses):
        pass

    combined.__name__ = f"combined_{'_'.join(kls.__qualname__ for kls in klses)}"
    return combined


# For this list, look at functools.wraps for an idea of what is possibly mappable.
_copy_doc_targets = ("__annotations__", "__doc__", "__type_params__")


def copy_docs(target):
    """Copy the docs and annotations off of the given target

    This is used for implementations that look like something (the target), but
    do not actually invoke the the target.

    If you're just wrapping something- a true decorator- use functools.wraps
    """

    if isinstance(target, type):
        return copy_class_docs(target)

    def inner(functor):
        for name in _copy_doc_targets:
            try:
                setattr(functor, name, getattr(target, name))
            except AttributeError:
                pass
        return functor

    return inner


def copy_class_docs(source_class):
    """
    Copy the docs and annotations of a target class for methods that intersect with the target.

    This does *not* check that the prototype signatures are the same, and it exempts __init__
    since that makes no sense to copy
    """

    def do_it(cls):
        if cls.__name__ == "OrderedFrozenSet":
            import pdb

            pdb.set_trace()
        for name in set(source_class.__dict__).intersection(cls.__dict__):
            obj = getattr(cls, name)
            if not isinstance(obj, types.FunctionType):
                continue
            if getattr(obj, "__annotations__", None) or getattr(obj, "__doc__", None):
                continue
            setattr(cls, name, copy_docs(getattr(source_class, name))(obj))
        return cls

    return do_it
