# Distributed under the terms of the GNU General Public License v2

EAPI=8

BRANCH_NAME="openssl-$(ver_cut 1-2)"

inherit flag-o-matic toolchain-funcs

DESCRIPTION="full-strength general purpose cryptography library (including SSL and TLS)"
HOMEPAGE="https://www.openssl.org/"
SNAPSHOT=3f8a5a4fdd28654dfe981ebd43d3dc1128a198c9
SRC_URI="https://github.com/openssl/openssl/archive/${SNAPSHOT}.tar.gz -> ${PN}-${SNAPSHOT}.tar.gz"
S=${WORKDIR}/openssl-${SNAPSHOT}

LICENSE="openssl"
SLOT="0"
KEYWORDS="amd64 arm64"

IUSE="static-libs test zlib"

RESTRICT="test"

DEPEND="
	zlib? ( lib-core/zlib )
"

BDEPEND="
	app-var/c_rehash
	app-lang/perl
	test? (
		app-core/diffutils
		app-util/bc
		app-core/procps
	)"

PATCHES=( "${FILESDIR}"/openssl-1.1.0j-parallel_install_fix.patch )

src_prepare() {
	append-flags -ffat-lto-objects -fno-ipa-sra
	filter-flags -fipa-pta
	# keep this in sync with app-var/c_rehash
	SSL_CNF_DIR="/etc/ssl"

	# Make sure we only ever touch Makefile.org and avoid patching a file
	# that gets blown away anyways by the Configure script in src_configure
	rm -f Makefile

	eapply_user

	sed -i \
    	-e '/^MANSUFFIX/s:=.*:=ssl:' \
    	-e '/^MAKEDEPPROG/s:=.*:=$(CC):' \
    	-e '/^install:/s:install_docs::' \
    	-e "/^DOCDIR/s@\$(BASENAME)@&-${PVR}@" \
    	Configurations/unix-Makefile.tmpl \
    	|| die


	# quiet out unknown driver argument warnings since openssl
	# doesn't have well-split CFLAGS and we're making it even worse
	# and 'make depend' uses -Werror for added fun (#417795 again)
	[[ ${CC} == *clang* ]] && append-flags -Qunused-arguments

	append-flags -fno-strict-aliasing
	append-flags $(test-flags-CC -Wa,--noexecstack)
	append-cppflags -DOPENSSL_NO_BUF_FREELISTS

	# Prefixify Configure shebang (#141906)
	sed \
		-e "1s,/usr/bin/env,${EPREFIX}&," \
		-i Configure || die
	# Remove test target when FEATURES=test isn't set
	if ! use test ; then
		sed \
			-e '/^$config{dirs}/s@ "test",@@' \
			-i Configure || die
	fi
	# The config script does stupid stuff to prompt the user.  Kill it.
	sed -i '/stty -icanon min 0 time 50; read waste/d' config || die
	./config --test-sanity || die "I AM NOT SANE"
}

src_configure() {
	unset APPS #197996
	unset SCRIPTS #312551
	unset CROSS_COMPILE #311473

	tc-export CC AR RANLIB RC

	if use arm64; then
		_target="linux-aarch64"
	fi

	if use amd64; then
		_target="linux-x86_64"
		_optflags="enable-ec_nistp_64_gcc_128"
	fi

	CFLAGS= LDFLAGS= \
	./Configure \
		$_target \
		$_optflags \
		${sslout} \
		no-ssl3 \
		no-ssl3-method \
		no-async \
		no-idea \
		no-mdc2 \
		no-rc5 \
		no-ec2m \
		no-sm2 \
		no-sm4 \
		no-seed \
		no-weak-ssl-ciphers \
		$(usex zlib enable-zlib no-zlib) \
		--prefix="${EPREFIX}"/usr \
		--libdir="${EPREFIX}"/usr/lib \
		--openssldir="${EPREFIX}"${SSL_CNF_DIR} \
		shared threads \
		|| die

	# Clean out hardcoded flags that openssl uses
	local DEFAULT_CFLAGS=$(grep ^CFLAGS= Makefile | LC_ALL=C sed \
		-e 's:^CFLAGS=::' \
		-e 's:\(^\| \)-fomit-frame-pointer::g' \
		-e 's:\(^\| \)-O[^ ]*::g' \
		-e 's:\(^\| \)-march=[^ ]*::g' \
		-e 's:\(^\| \)-mcpu=[^ ]*::g' \
		-e 's:\(^\| \)-m[^ ]*::g' \
		-e 's:^ *::' \
		-e 's: *$::' \
		-e 's: \+: :g' \
		-e 's:\\:\\\\:g'
	)

	# Now insert clean default flags with user flags
	sed -i \
		-e "/^CFLAGS=/s|=.*|=${DEFAULT_CFLAGS} ${CFLAGS}|" \
		-e "/^LDFLAGS=/s|=[[:space:]]*$|=${LDFLAGS}|" \
		Makefile || die
}

src_compile() {
	# depend is needed to use $confopts; it also doesn't matter
	# that it's -j1 as the code itself serializes subdirs
	emake -j1 depend
	emake all
}

src_test() {
	emake -j1 test
}

src_install() {
	# We need to create $ED/usr on our own to avoid a race condition #665130
	if [[ ! -d "${ED}/usr" ]]; then
		# We can only create this directory once
		mkdir "${ED}"/usr || die
	fi

	default

	# This is crappy in that the static archives are still built even
	# when USE=static-libs.  But this is due to a failing in the openssl
	# build system: the static archives are built as PIC all the time.
	# Only way around this would be to manually configure+compile openssl
	# twice; once with shared lib support enabled and once without.
	use static-libs || rm "${ED}"/usr/lib/{libssl,libcrypto}.a || die

	# create the certs directory
	keepdir ${SSL_CNF_DIR}/certs

	dodir /etc/sandbox.d #254521
	echo 'SANDBOX_PREDICT="/dev/crypto"' > "${ED}"/etc/sandbox.d/10openssl

	diropts -m0700
	keepdir ${SSL_CNF_DIR}/private

	rm "${ED}"/usr/bin/c_rehash
	rm -rf "${ED}"/usr/share/man
}

pkg_postinst() {
	ebegin "Running 'c_rehash ${EROOT}${SSL_CNF_DIR}/certs/' to rebuild hashes"
	c_rehash "${EROOT}${SSL_CNF_DIR}/certs" >/dev/null
	eend $?
}
