// SPDX-FileCopyrightText: 2011 Mathieu Desnoyers <mathieu.desnoyers@efficios.com>
// SPDX-FileCopyrightText: 2011 Lai Jiangshan <laijs@cn.fujitsu.com>
//
// SPDX-License-Identifier: LGPL-2.1-or-later

#ifndef _URCU_RCULFHASH_H
#define _URCU_RCULFHASH_H

/*
 * Userspace RCU library - Lock-Free RCU Hash Table
 *
 * For use with URCU_API_MAP (API mapping of liburcu), include this file
 * _after_ including your URCU flavor.
 */

#include <urcu/config.h>
#include <stdint.h>
#include <pthread.h>
#include <urcu/compiler.h>

#ifdef __cplusplus
extern "C" {
#endif

struct cds_lfht;

/*
 * cds_lfht_node: Contains the next pointers and reverse-hash
 * value required for lookup and traversal of the hash table.
 *
 * struct cds_lfht_node should be aligned on 8-bytes boundaries because
 * the three lower bits are used as flags. It is worth noting that the
 * information contained within these three bits could be represented on
 * two bits by re-using the same bit for REMOVAL_OWNER_FLAG and
 * BUCKET_FLAG. This can be done if we ensure that no iterator nor
 * updater check the BUCKET_FLAG after it detects that the REMOVED_FLAG
 * is set. Given the minimum size of struct cds_lfht_node is 8 bytes on
 * 32-bit architectures, we choose to go for simplicity and reserve
 * three bits.
 *
 * struct cds_lfht_node can be embedded into a structure (as a field).
 * caa_container_of() can be used to get the structure from the struct
 * cds_lfht_node after a lookup.
 *
 * The structure which embeds it typically holds the key (or key-value
 * pair) of the object. The caller code is responsible for calculation
 * of the hash value for cds_lfht APIs.
 */
struct cds_lfht_node {
	struct cds_lfht_node *next;	/* ptr | REMOVAL_OWNER_FLAG | BUCKET_FLAG | REMOVED_FLAG */
	unsigned long reverse_hash;
} __attribute__((__aligned__(8)));

/* cds_lfht_iter: Used to track state while traversing a hash chain. */
struct cds_lfht_iter {
	struct cds_lfht_node *node, *next;
	/*
	 * For debugging purposes, build both API users and rculfhash
	 * library with CDS_LFHT_ITER_DEBUG defined. This enables extra
	 * consistency checks for calls to a cds_lfht_next() or
	 * cds_lfht_next_duplicate() after the iterator has been
	 * re-purposed to iterate on a different hash table. This is a
	 * common programming mistake when performing hash table lookup
	 * nested in a hash table traversal.
	 */
#ifdef CONFIG_CDS_LFHT_ITER_DEBUG
	struct cds_lfht *lfht;
#endif
};

/*
 * cds_lfht_alloc: Callbacks if we want to use custom memory allocator.
 */
struct cds_lfht_alloc {
	void *(*malloc)(void *state, size_t size);
	void *(*calloc)(void *state, size_t nmemb, size_t size);
	void *(*realloc)(void *state, void *ptr, size_t size);
	void *(*aligned_alloc)(void *state, size_t alignment, size_t size);
	void  (*free)(void *state, void *ptr);
	void  *state;
};

static inline
struct cds_lfht_node *cds_lfht_iter_get_node(const struct cds_lfht_iter *iter)
{
	return iter->node;
}

struct rcu_flavor_struct;

/*
 * Caution !
 * Ensure reader and writer threads are registered as urcu readers.
 */

typedef int (*cds_lfht_match_fct)(struct cds_lfht_node *node, const void *key);

/*
 * cds_lfht_node_init - Initialize a hash table node
 * @node: The node to initialize.
 *
 * This function is kept to be eventually used for debugging purposes
 * (detection of memory corruption).
 */
static inline
void cds_lfht_node_init(struct cds_lfht_node *node __attribute__((__unused__)))
{
}

/*
 * cds_lfht_node_init_deleted - Initialize a hash table node to "removed" state
 * @node: The node to initialize.
 *
 * Initialize the node such that cds_lfht_is_node_deleted() can be used
 * on the node before it is added to a hash table.
 */
extern
void cds_lfht_node_init_deleted(struct cds_lfht_node *node);

/*
 * Hash table creation flags.
 */
enum {
	CDS_LFHT_AUTO_RESIZE = (1U << 0),
	CDS_LFHT_ACCOUNTING = (1U << 1),
};

struct cds_lfht_mm_type {
	struct cds_lfht *(*alloc_cds_lfht)(unsigned long min_nr_alloc_buckets,
			unsigned long max_nr_buckets, const struct cds_lfht_alloc *alloc);
	void (*alloc_bucket_table)(struct cds_lfht *ht, unsigned long order);
	void (*free_bucket_table)(struct cds_lfht *ht, unsigned long order);
	struct cds_lfht_node *(*bucket_at)(struct cds_lfht *ht,
			unsigned long index);
};

extern const struct cds_lfht_mm_type cds_lfht_mm_order;
extern const struct cds_lfht_mm_type cds_lfht_mm_chunk;
extern const struct cds_lfht_mm_type cds_lfht_mm_mmap;

/*
 * _cds_lfht_new - API used by cds_lfht_new wrapper. Do not use directly.
 */
extern
struct cds_lfht *_cds_lfht_new(unsigned long init_size,
			unsigned long min_nr_alloc_buckets,
			unsigned long max_nr_buckets,
			int flags,
			const struct cds_lfht_mm_type *mm,
			const struct rcu_flavor_struct *flavor,
			pthread_attr_t *attr);

/*
 * _cds_lfht_new_with_alloc - API used by cds_lfht_new_with_flavor_alloc.
 */
extern
struct cds_lfht *_cds_lfht_new_with_alloc(unsigned long init_size,
			unsigned long min_nr_alloc_buckets,
			unsigned long max_nr_buckets,
			int flags,
			const struct cds_lfht_mm_type *mm,
			const struct rcu_flavor_struct *flavor,
			const struct cds_lfht_alloc *alloc,
			pthread_attr_t *attr);

/*
 * cds_lfht_new_flavor - Allocate a hash table tied to a RCU flavor.
 * @init_size: Number of buckets to allocate and initialize (chain into
 *             the hash table) at hash table creation. The resize target
 *             is initialized to this value. Must a be a power of two.
 * @min_nr_alloc_buckets: The minimum number of allocated buckets.
 *                        It acts as a lower bound below which bucket
 *                        memory will not be freed, but the buckets are
 *                        still unlinked from the hash table.
 *                        Must be a power of two. This parameter only
 *                        affects the hash table when an explicit
 *                        cds_lfht_resize, or cds_lfht_resize_lazy_count
 *                        are used, or when the hash table is created
 *                        with the CDS_LFHT_AUTO_RESIZE flag.
 *                        It is recommended to use an
 *                        @init_size <= @min_nr_alloc_buckets for
 *                        auto-resized hash tables.
 * @max_nr_buckets: The maximum number of hash table buckets allowed.
 *                  (must be a power of two, 0 is accepted, means
 *                  "infinite")
 * @flags: Hash table creation flags (can be combined with bitwise or: '|').
 *           0: no flags.
 *           CDS_LFHT_AUTO_RESIZE: Automatically resize the hash table.
 *           CDS_LFHT_ACCOUNTING: Count the number of node addition
 *                                and removal in the table.
 * @flavor: Flavor of liburcu to use to synchronize the hash table
 * @attr: Optional resize worker thread attributes. NULL for default.
 *
 * Return NULL on error.
 * Note: the RCU flavor must be already included before the hash table header.
 *
 * The programmer is responsible for ensuring that resize operation has a
 * priority equal to hash table updater threads. It should be performed by
 * specifying the appropriate priority in the pthread "attr" argument, and,
 * for CDS_LFHT_AUTO_RESIZE, by ensuring that call_rcu worker threads also have
 * this priority level. Having lower priority for call_rcu and resize threads
 * does not pose any correctness issue, but the resize operations could be
 * starved by updates, thus leading to long hash table bucket chains.
 * Threads calling cds_lfht_new are NOT required to be registered RCU
 * read-side threads. It can be called very early. (e.g. before RCU is
 * initialized)
 */
static inline
struct cds_lfht *cds_lfht_new_flavor(unsigned long init_size,
			unsigned long min_nr_alloc_buckets,
			unsigned long max_nr_buckets,
			int flags,
			const struct rcu_flavor_struct *flavor,
			pthread_attr_t *attr)
{
	return _cds_lfht_new(init_size, min_nr_alloc_buckets, max_nr_buckets,
			flags, NULL, flavor, attr);
}

/*
 * cds_lfht_new_with_flavor_alloc - Allocate a hash table tied to a RCU flavor.
 * @init_size: Number of buckets to allocate and initialize (chain into
 *             the hash table) at hash table creation. The resize target
 *             is initialized to this value. Must be a power of two.
 * @min_nr_alloc_buckets: The minimum number of allocated buckets.
 *                        It acts as a lower bound below which bucket
 *                        memory will not be freed, but the buckets are
 *                        still unlinked from the hash table.
 *                        Must be a power of two. This parameter only
 *                        affects the hash table when an explicit
 *                        cds_lfht_resize, or cds_lfht_resize_lazy_count
 *                        are used, or when the hash table is created
 *                        with the CDS_LFHT_AUTO_RESIZE flag.
 *                        It is recommended to use an
 *                        @init_size <= @min_nr_alloc_buckets for
 *                        auto-resized hash tables.
 * @max_nr_buckets: The maximum number of hash table buckets allowed.
 *                  (must be a power of two, 0 is accepted, means
 *                  "infinite")
 * @flags: Hash table creation flags (can be combined with bitwise or: '|').
 *           0: no flags.
 *           CDS_LFHT_AUTO_RESIZE: Automatically resize the hash table.
 *           CDS_LFHT_ACCOUNTING: Count the number of node addition
 *                                and removal in the table.
 * @flavor: Flavor of liburcu to use to synchronize the hash table
 * @alloc: Custom memory allocator for hash table memory management.
 *         NULL for default. If a custom allocator is used, then
 *         the whole interface of struct cds_lfht_alloc must be implemented.
 * @attr: Optional resize worker thread attributes. NULL for default.
 *
 * Return NULL on error.
 * Note: the RCU flavor must be already included before the hash table header.
 *
 * The programmer is responsible for ensuring that resize operation has a
 * priority equal to hash table updater threads. It should be performed by
 * specifying the appropriate priority in the pthread "attr" argument, and,
 * for CDS_LFHT_AUTO_RESIZE, by ensuring that call_rcu worker threads also have
 * this priority level. Having lower priority for call_rcu and resize threads
 * does not pose any correctness issue, but the resize operations could be
 * starved by updates, thus leading to long hash table bucket chains.
 * Threads calling cds_lfht_new are NOT required to be registered RCU
 * read-side threads. It can be called very early. (e.g. before RCU is
 * initialized)
 */
static inline
struct cds_lfht *cds_lfht_new_with_flavor_alloc(unsigned long init_size,
			unsigned long min_nr_alloc_buckets,
			unsigned long max_nr_buckets,
			int flags,
			const struct rcu_flavor_struct *flavor,
			const struct cds_lfht_alloc *alloc,
			pthread_attr_t *attr)
{
	return _cds_lfht_new_with_alloc(init_size, min_nr_alloc_buckets, max_nr_buckets,
			flags, NULL, flavor, alloc, attr);
}


#ifdef URCU_API_MAP
/*
 * cds_lfht_new - Allocate a hash table.
 * @init_size: Number of buckets to allocate and initialize (chain into
 *             the hash table) at hash table creation. The resize target
 *             is initialized to this value. Must be a power of two.
 * @min_nr_alloc_buckets: The minimum number of allocated buckets.
 *                        It acts as a lower bound below which bucket
 *                        memory will not be freed, but the buckets are
 *                        still unlinked from the hash table.
 *                        Must be a power of two. This parameter only
 *                        affects the hash table when an explicit
 *                        cds_lfht_resize, or cds_lfht_resize_lazy_count
 *                        are used, or when the hash table is created
 *                        with the CDS_LFHT_AUTO_RESIZE flag.
 *                        It is recommended to use an
 *                        @init_size <= @min_nr_alloc_buckets for
 *                        auto-resized hash tables.
 * @max_nr_buckets: The maximum number of hash table buckets allowed.
 *                  (must be a power of two, 0 is accepted, means
 *                  "infinite")
 * @flags: Hash table creation flags (can be combined with bitwise or: '|').
 *           0: no flags.
 *           CDS_LFHT_AUTO_RESIZE: Automatically resize the hash table.
 *           CDS_LFHT_ACCOUNTING: Count the number of node addition
 *                                and removal in the table.
 * @attr: Optional resize worker thread attributes. NULL for default.
 *
 * Return NULL on error.
 * Note: the RCU flavor must be already included before the hash table header.
 *
 * The programmer is responsible for ensuring that resize operation has a
 * priority equal to hash table updater threads. It should be performed by
 * specifying the appropriate priority in the pthread "attr" argument, and,
 * for CDS_LFHT_AUTO_RESIZE, by ensuring that call_rcu worker threads also have
 * this priority level. Having lower priority for call_rcu and resize threads
 * does not pose any correctness issue, but the resize operations could be
 * starved by updates, thus leading to long hash table bucket chains.
 * Threads calling cds_lfht_new are NOT required to be registered RCU
 * read-side threads. It can be called very early. (e.g. before RCU is
 * initialized)
 */
static inline
struct cds_lfht *cds_lfht_new(unsigned long init_size,
			unsigned long min_nr_alloc_buckets,
			unsigned long max_nr_buckets,
			int flags,
			pthread_attr_t *attr)
{
	return _cds_lfht_new(init_size, min_nr_alloc_buckets, max_nr_buckets,
			flags, NULL, &rcu_flavor, attr);
}
#endif /* URCU_API_MAP */

/*
 * cds_lfht_destroy - Destroy a hash table.
 * @ht: The hash table to destroy.
 * @attr: (output) Resize worker thread attributes, as received by cds_lfht_new.
 *        The caller will typically want to free this pointer if dynamically
 *        allocated. The attr point can be NULL if the caller does not
 *        need to be informed of the value passed to cds_lfht_new().
 *
 * Return 0 on success, negative error value on error.

 * Threads calling this API need to be registered RCU read-side threads.
 *
 * Prior to liburcu 0.10:
 * - cds_lfht_destroy should *not* be called from a RCU read-side
 *   critical section. It should *not* be called from a call_rcu thread
 *   context neither.
 *
 * Starting from liburcu 0.10, rculfhash implements its own worker
 * thread to handle resize operations, which removes the above RCU
 * read-side critical section requirement on cds_lfht_destroy.
 */
extern
int cds_lfht_destroy(struct cds_lfht *ht, pthread_attr_t **attr);

/*
 * cds_lfht_count_nodes - Count the number of nodes in the hash table.
 * @ht: The hash table.
 * @split_count_before: Sample the node count split-counter before traversal.
 * @count: Traverse the hash table, count the number of nodes observed.
 * @split_count_after: Sample the node count split-counter after traversal.
 *
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 */
extern
void cds_lfht_count_nodes(struct cds_lfht *ht,
		long *split_count_before,
		unsigned long *count,
		long *split_count_after);

/*
 * cds_lfht_lookup - Lookup a node by key.
 * @ht: The hash table.
 * @hash: The key hash.
 * @match: The key match function.
 * @key: The current node key.
 * @iter: Node, if found (output). *iter->node set to NULL if not found.
 *
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * This function acts as a rcu_dereference() to read the node pointer.
 */
extern
void cds_lfht_lookup(struct cds_lfht *ht, unsigned long hash,
		cds_lfht_match_fct match, const void *key,
		struct cds_lfht_iter *iter);

/*
 * cds_lfht_next_duplicate - Get the next item with same key, after iterator.
 * @ht: The hash table.
 * @match: The key match function.
 * @key: The current node key.
 * @iter: Input: Current iterator.
 *        Output: Node, if found. *iter->node set to NULL if not found.
 *
 * Uses an iterator initialized by a lookup or traversal. Important: the
 * iterator _needs_ to be initialized before calling
 * cds_lfht_next_duplicate.
 * Sets *iter-node to the following node with same key.
 * Sets *iter->node to NULL if no following node exists with same key.
 * RCU read-side lock must be held across cds_lfht_lookup and
 * cds_lfht_next calls, and also between cds_lfht_next calls using the
 * node returned by a previous cds_lfht_next.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * This function acts as a rcu_dereference() to read the node pointer.
 */
extern
void cds_lfht_next_duplicate(struct cds_lfht *ht,
		cds_lfht_match_fct match, const void *key,
		struct cds_lfht_iter *iter);

/*
 * cds_lfht_first - Get the first node in the table.
 * @ht: The hash table.
 * @iter: First node, if exists (output). *iter->node set to NULL if not found.
 *
 * Output in "*iter". *iter->node set to NULL if table is empty.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * This function acts as a rcu_dereference() to read the node pointer.
 */
extern
void cds_lfht_first(struct cds_lfht *ht, struct cds_lfht_iter *iter);

/*
 * cds_lfht_next - Get the next node in the table.
 * @ht: The hash table.
 * @iter: Input: Current iterator.
 *        Output: Next node, if exists. *iter->node set to NULL if not found.
 *
 * Input/Output in "*iter". *iter->node set to NULL if *iter was
 * pointing to the last table node.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * This function acts as a rcu_dereference() to read the node pointer.
 */
extern
void cds_lfht_next(struct cds_lfht *ht, struct cds_lfht_iter *iter);

/*
 * cds_lfht_add - Add a node to the hash table.
 * @ht: The hash table.
 * @hash: The key hash.
 * @node: The node to add.
 *
 * This function supports adding redundant keys into the table.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * This function issues a full memory barrier before and after its
 * atomic commit.
 */
extern
void cds_lfht_add(struct cds_lfht *ht, unsigned long hash,
		struct cds_lfht_node *node);

/*
 * cds_lfht_add_unique - Add a node to hash table, if key is not present.
 * @ht: The hash table.
 * @hash: The node's hash.
 * @match: The key match function.
 * @key: The node's key.
 * @node: The node to try adding.
 *
 * Return the node added upon success.
 * Return the unique node already present upon failure. If
 * cds_lfht_add_unique fails, the node passed as parameter should be
 * freed by the caller. In this case, the caller does NOT need to wait
 * for a grace period before freeing or re-using the node.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 *
 * The semantic of this function is that if only this function is used
 * to add keys into the table, no duplicated keys should ever be
 * observable in the table. The same guarantee apply for combination of
 * add_unique and add_replace (see below).
 *
 * Upon success, this function issues a full memory barrier before and
 * after its atomic commit. Upon failure, this function acts like a
 * simple lookup operation: it acts as a rcu_dereference() to read the
 * node pointer. The failure case does not guarantee any other memory
 * barrier.
 */
extern
struct cds_lfht_node *cds_lfht_add_unique(struct cds_lfht *ht,
		unsigned long hash,
		cds_lfht_match_fct match,
		const void *key,
		struct cds_lfht_node *node);

/*
 * cds_lfht_add_replace - Replace or add a node within hash table.
 * @ht: The hash table.
 * @hash: The node's hash.
 * @match: The key match function.
 * @key: The node's key.
 * @node: The node to add.
 *
 * Return the node replaced upon success. If no node matching the key
 * was present, return NULL, which also means the operation succeeded.
 * This replacement operation should never fail.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * After successful replacement, a grace period must be waited for before
 * freeing or re-using the memory reserved for the returned node.
 *
 * The semantic of replacement vs lookups and traversals is the
 * following: if lookups and traversals are performed between a key
 * unique insertion and its removal, we guarantee that the lookups and
 * traversals will always find exactly one instance of the key if it is
 * replaced concurrently with the lookups.
 *
 * Providing this semantic allows us to ensure that replacement-only
 * schemes will never generate duplicated keys. It also allows us to
 * guarantee that a combination of add_replace and add_unique updates
 * will never generate duplicated keys.
 *
 * This function issues a full memory barrier before and after its
 * atomic commit.
 */
extern
struct cds_lfht_node *cds_lfht_add_replace(struct cds_lfht *ht,
		unsigned long hash,
		cds_lfht_match_fct match,
		const void *key,
		struct cds_lfht_node *node);

/*
 * cds_lfht_replace - Replace a node pointed to by iter within hash table.
 * @ht: The hash table.
 * @old_iter: The iterator position of the node to replace.
 * @hash: The node's hash.
 * @match: The key match function.
 * @key: The node's key.
 * @new_node: The new node to use as replacement.
 *
 * Return 0 if replacement is successful, negative value otherwise.
 * Replacing a NULL old node or an already removed node will fail with
 * -ENOENT.
 * If the hash or value of the node to replace and the new node differ,
 * this function returns -EINVAL without proceeding to the replacement.
 * Old node can be looked up with cds_lfht_lookup and cds_lfht_next.
 * RCU read-side lock must be held between lookup and replacement.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * After successful replacement, a grace period must be waited for before
 * freeing or re-using the memory reserved for the old node (which can
 * be accessed with cds_lfht_iter_get_node).
 *
 * The semantic of replacement vs lookups is the same as
 * cds_lfht_add_replace().
 *
 * Upon success, this function issues a full memory barrier before and
 * after its atomic commit. Upon failure, this function does not issue
 * any memory barrier.
 */
extern
int cds_lfht_replace(struct cds_lfht *ht,
		struct cds_lfht_iter *old_iter,
		unsigned long hash,
		cds_lfht_match_fct match,
		const void *key,
		struct cds_lfht_node *new_node);

/*
 * cds_lfht_del - Remove node pointed to by iterator from hash table.
 * @ht: The hash table.
 * @node: The node to delete.
 *
 * Return 0 if the node is successfully removed, negative value
 * otherwise.
 * Deleting a NULL node or an already removed node will fail with a
 * negative value.
 * Node can be looked up with cds_lfht_lookup and cds_lfht_next,
 * followed by use of cds_lfht_iter_get_node.
 * RCU read-side lock must be held between lookup and removal.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * After successful removal, a grace period must be waited for before
 * freeing or re-using the memory reserved for old node (which can be
 * accessed with cds_lfht_iter_get_node).
 * Upon success, this function issues a full memory barrier before and
 * after its atomic commit. Upon failure, this function does not issue
 * any memory barrier.
 */
extern
int cds_lfht_del(struct cds_lfht *ht, struct cds_lfht_node *node);

/*
 * cds_lfht_is_node_deleted - Query whether a node is removed from hash table.
 *
 * Return non-zero if the node is deleted from the hash table, 0
 * otherwise.
 * Node can be looked up with cds_lfht_lookup and cds_lfht_next,
 * followed by use of cds_lfht_iter_get_node.
 * RCU read-side lock must be held between lookup and call to this
 * function.
 * Call with rcu_read_lock held.
 * Threads calling this API need to be registered RCU read-side threads.
 * This function does not issue any memory barrier.
 */
extern
int cds_lfht_is_node_deleted(const struct cds_lfht_node *node);

/*
 * cds_lfht_resize - Force a hash table resize
 * @ht: The hash table.
 * @new_size: Update to this hash table size.
 *
 * Threads calling this API need to be registered RCU read-side threads.
 * This function does not (necessarily) issue memory barriers.
 * cds_lfht_resize should *not* be called from a RCU read-side critical
 * section.
 */
extern
void cds_lfht_resize(struct cds_lfht *ht, unsigned long new_size);

/*
 * Note: it is safe to perform element removal (del), replacement, or
 * any hash table update operation during any of the following hash
 * table traversals.
 * These functions act as rcu_dereference() to read the node pointers.
 */
#define cds_lfht_for_each(ht, iter, node)				\
	for (cds_lfht_first(ht, iter),					\
			node = cds_lfht_iter_get_node(iter);		\
		node != NULL;						\
		cds_lfht_next(ht, iter),				\
			node = cds_lfht_iter_get_node(iter))

#define cds_lfht_for_each_duplicate(ht, hash, match, key, iter, node)	\
	for (cds_lfht_lookup(ht, hash, match, key, iter),		\
			node = cds_lfht_iter_get_node(iter);		\
		node != NULL;						\
		cds_lfht_next_duplicate(ht, match, key, iter),		\
			node = cds_lfht_iter_get_node(iter))

#define cds_lfht_entry(ptr, type, member)				\
	caa_container_of_check_null(ptr, type, member)

#define cds_lfht_for_each_entry(ht, iter, pos, member)			\
	for (cds_lfht_first(ht, iter),					\
			pos = cds_lfht_entry(cds_lfht_iter_get_node(iter), \
				__typeof__(*(pos)), member);		\
		pos != NULL;						\
		cds_lfht_next(ht, iter),				\
			pos = cds_lfht_entry(cds_lfht_iter_get_node(iter), \
				__typeof__(*(pos)), member))

#define cds_lfht_for_each_entry_duplicate(ht, hash, match, key,		\
				iter, pos, member)			\
	for (cds_lfht_lookup(ht, hash, match, key, iter),		\
			pos = cds_lfht_entry(cds_lfht_iter_get_node(iter), \
				__typeof__(*(pos)), member);		\
		pos != NULL;						\
		cds_lfht_next_duplicate(ht, match, key, iter),		\
			pos = cds_lfht_entry(cds_lfht_iter_get_node(iter), \
				__typeof__(*(pos)), member))

#ifdef __cplusplus
}
#endif

#endif /* _URCU_RCULFHASH_H */
