# Distributed under the terms of the GNU General Public License v2

EAPI=8

inherit python-any-r1

DESCRIPTION="Common CA Certificates PEM files"
HOMEPAGE="https://packages.debian.org/sid/ca-certificates"
SNAPSHOT=ba3830faf6207f6444827209915dcfc4ce44b272
SRC_URI="https://salsa.debian.org/debian/ca-certificates/-/archive/${SNAPSHOT}/ca-certificates-${SNAPSHOT}.tar.bz2"
S="${WORKDIR}/ca-certificates-${SNAPSHOT}"

LICENSE="MPL-1.1"
RESTRICT="network-sandbox"

SLOT="0"
KEYWORDS="amd64 arm64"

DEPEND="
app-var/c_rehash
app-core/debianutils
"

pkg_setup() {
	# For the conversion to having it in CONFIG_PROTECT_MASK,
	# we need to tell users about it once manually first.
	[[ -f "${EPREFIX}"/etc/env.d/98ca-certificates ]] \
		|| ewarn "You should run update-ca-certificates manually after etc-update"

	python-any-r1_pkg_setup
}

src_prepare() {
	# Try to fetch, fallback to FILESDIR if it fails
	if ! wget -O "${S}/mozilla/certdata.txt" "https://hg.mozilla.org/releases/mozilla-release/raw-file/default/security/nss/lib/ckfw/builtins/certdata.txt"; then
    	if [[ -f "${FILESDIR}/certdata.txt" ]]; then
        	einfo "Falling back to ${FILESDIR}/certdata.txt"
        	cp "${FILESDIR}/certdata.txt" "${S}/mozilla/certdata.txt" || die
    	else
        	die "Failed to fetch certdata.txt and no fallback in \${FILESDIR}"
    	fi
	fi

	if ! wget -O "${S}/mozilla/nssckbi.h" "https://hg.mozilla.org/releases/mozilla-release/raw-file/default/security/nss/lib/ckfw/builtins/nssckbi.h"; then
    	if [[ -f "${FILESDIR}/nssckbi.h" ]]; then
        	einfo "Falling back to ${FILESDIR}/nssckbi.h"
        	cp "${FILESDIR}/nssckbi.h" "${S}/mozilla/nssckbi.h" || die
    	else
        	die "Failed to fetch nssckbi.h and no fallback in \${FILESDIR}"
    	fi
	fi

	default
	cp "${FILESDIR}/certdata2pem.py" "${S}/mozilla/certdata2pem.py" || die
}

src_compile() {
	emake -C "mozilla"
}

src_install() {
	keepdir /etc/ssl/certs/
    dobin "${S}/sbin/update-ca-certificates"
    doman "${S}/sbin/update-ca-certificates.8"

	for f in "${S}"/mozilla/*.crt; do
    	echo "Checking $f ..."
    	enddate=$(openssl x509 -enddate -noout -in "$f" | cut -d= -f2)
    	echo "  End date: $enddate"
    	if ! enddate_epoch=$(date -u -d "$enddate" +%s 2>/dev/null); then
        	echo "  [WARN] Could not parse end date for $f. Skipping."
        	continue
    	fi
    	now_epoch=$(date -u +%s)
    	echo "  End date (epoch): $enddate_epoch"
    	echo "  Now      (epoch): $now_epoch"
    	if [ "$enddate_epoch" -lt "$now_epoch" ]; then
        	echo "  [EXPIRE] Deleting expired cert: $f"
        	rm "$f"
    	else
        	echo "  [OK] Not expired."
    	fi
	done

	insinto /usr/share/ca-certificates/mozilla
    doins "${S}"/mozilla/*.crt

    dodir /etc
    {
        echo "# Automatically generated by ${CATEGORY}/${PF}"
        echo "# $(date -u)"
        echo "# Do not edit."

        find "${ED}/usr/share/ca-certificates" -type f -name '*.crt' \
            | sed "s|^${ED}/usr/share/ca-certificates/||" \
            | LC_ALL=C sort
    } > "${ED}/etc/ca-certificates.conf" || die "Failed to generate ca-certificates.conf"
}

pkg_postinst() {
	rm -rf /etc/ssl/certs/*
	update-ca-certificates
}
